"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.backendToUiReportsList = exports.backendToUiReportDefinitionsList = exports.backendToUiReportDefinition = exports.backendToUiReport = void 0;
var _model = require("../../../model");
var _backendModel = require("../../../model/backendModel");
var _constants = require("../constants");
var _moment = _interopRequireDefault(require("moment"));
var _url = require("url");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const backendToUiReport = (backendReportInstance, serverBasePath) => {
  const {
    inContextDownloadUrlPath,
    beginTimeMs,
    endTimeMs,
    tenant,
    status,
    lastUpdatedTimeMs: reportLastUpdatedTimeMs,
    createdTimeMs: reportCreatedTimeMs,
    reportDefinitionDetails: backendReportDefinitionDetails
  } = backendReportInstance;
  const {
    reportDefinition: {
      source: {
        type: sourceType,
        id: sourceId
      },
      delivery
    }
  } = backendReportDefinitionDetails;
  const baseUrl = getBaseUrl(sourceType, sourceId);
  let report = {
    // inContextDownloadUrlPath may not exist for report instance created from scheduled job
    query_url: inContextDownloadUrlPath || getUiQueryUrl(baseUrl, beginTimeMs, endTimeMs, tenant),
    time_from: beginTimeMs,
    time_to: endTimeMs,
    last_updated: reportLastUpdatedTimeMs,
    time_created: reportCreatedTimeMs,
    state: getUiReportState(status, delivery),
    report_definition: backendToUiReportDefinition(backendReportDefinitionDetails, serverBasePath)
  };

  // Add severbasePath back to query_url
  report.query_url = serverBasePath + report.query_url;
  return report;
};
exports.backendToUiReport = backendToUiReport;
const backendToUiReportsList = (backendReportsList, serverBasePath) => {
  const res = backendReportsList.map(backendReport => {
    return {
      _id: backendReport.id,
      _source: backendToUiReport(backendReport, serverBasePath)
    };
  });
  return res;
};
exports.backendToUiReportsList = backendToUiReportsList;
const backendToUiReportDefinition = (backendReportDefinitionDetails, serverBasePath) => {
  const {
    lastUpdatedTimeMs,
    createdTimeMs,
    reportDefinition: {
      name,
      isEnabled,
      source: {
        type: sourceType,
        description,
        id: sourceId,
        origin
      },
      format: {
        fileFormat,
        duration,
        header,
        footer,
        limit,
        timeFrom: timeFrom,
        timeTo: timeTo
      },
      trigger: {
        triggerType,
        schedule
      },
      delivery
    }
  } = backendReportDefinitionDetails;
  const baseUrl = getBaseUrl(sourceType, sourceId);
  const reportSource = getUiReportSource(sourceType);
  let uiReportDefinition = {
    report_params: {
      report_name: name,
      report_source: reportSource,
      description: description,
      core_params: reportSource === _constants.REPORT_TYPE.savedSearch ? getDataReportCoreParams(limit, sourceId, fileFormat, duration, baseUrl, origin, timeFrom, timeTo) : getVisualReportCoreParams(fileFormat, header, footer, duration, baseUrl, origin, timeFrom, timeTo)
    },
    trigger: getUiTriggerParams(triggerType, schedule, createdTimeMs, isEnabled),
    delivery: getUiDeliveryParams(delivery),
    //TODO:
    time_created: createdTimeMs,
    last_updated: lastUpdatedTimeMs,
    status: getUiReportDefinitionStatus(isEnabled)
  };
  // validate to assign default values to some fields for UI model
  uiReportDefinition = _model.reportDefinitionSchema.validate(uiReportDefinition);
  uiReportDefinition.report_params.core_params.base_url = serverBasePath + uiReportDefinition.report_params.core_params.base_url;
  return uiReportDefinition;
};
exports.backendToUiReportDefinition = backendToUiReportDefinition;
const backendToUiReportDefinitionsList = (backendReportDefinitionDetailsList, serverBasePath) => {
  const res = backendReportDefinitionDetailsList.map(backendReportDefinitionDetails => {
    return {
      _id: backendReportDefinitionDetails.id,
      _source: {
        // TODO: this property can be removed, but need UI changes as well
        report_definition: backendToUiReportDefinition(backendReportDefinitionDetails, serverBasePath)
      }
    };
  });
  return res;
};
exports.backendToUiReportDefinitionsList = backendToUiReportDefinitionsList;
const getVisualReportCoreParams = (fileFormat, header = '', footer = '', duration, baseUrl, origin, timeFrom = null, timeTo = null) => {
  let res = {
    base_url: baseUrl,
    report_format: getUiReportFormat(fileFormat),
    header: header,
    footer: footer,
    time_duration: duration,
    origin: origin,
    timeFrom: timeFrom !== null && timeFrom !== void 0 ? timeFrom : undefined,
    timeTo: timeTo !== null && timeTo !== void 0 ? timeTo : undefined
  };
  return res;
};

// queryUrl = baseUrl + time range
const getUiQueryUrl = (baseUrl, beginTimeMs, endTimeMs, tenant) => {
  const timeFrom = (0, _moment.default)(beginTimeMs).toISOString();
  const timeTo = (0, _moment.default)(endTimeMs).toISOString();
  let queryUrl = `${baseUrl}?_g=(time:(from:'${timeFrom}',to:'${timeTo}'))`;
  if (tenant !== undefined) {
    if (tenant === '') {
      tenant = 'global';
    } else if (tenant === '__user__') {
      tenant = 'private';
    }
    queryUrl = addTenantToURL(queryUrl, tenant);
  }
  return queryUrl;
};
const getBaseUrl = (sourceType, sourceId) => {
  //TODO: AES domain has different prefix, need figure out a general solution
  const baseUrl = `${_backendModel.URL_PREFIX_DICT[sourceType]}${sourceId}`;
  return baseUrl;
};
const getDataReportCoreParams = (limit = _constants.DEFAULT_MAX_SIZE, sourceId, fileFormat, duration, baseUrl, origin, timeFrom = null, timeTo = null) => {
  let res = {
    base_url: baseUrl,
    report_format: getUiReportFormat(fileFormat),
    limit: limit,
    time_duration: duration,
    saved_search_id: sourceId,
    origin: origin,
    timeFrom: timeFrom !== null && timeFrom !== void 0 ? timeFrom : undefined,
    timeTo: timeTo !== null && timeTo !== void 0 ? timeTo : undefined
  };
  return res;
};
const getUiScheduleParams = (schedule, createdTimeMs, isEnabled) => {
  let res = {
    trigger_params: {
      enabled_time: createdTimeMs,
      enabled: isEnabled,
      schedule_type: schedule && 'cron' in schedule ? _constants.SCHEDULE_TYPE.cron : _constants.SCHEDULE_TYPE.recurring,
      schedule: schedule
    }
  };
  return res;
};
const getUiTriggerType = backendField => {
  let res;
  for (let [ui, backendFieldList] of Object.entries(_backendModel.TRIGGER_TYPE_DICT)) {
    for (let item of backendFieldList) {
      if (item === backendField) {
        res = ui;
      }
    }
  }
  return res;
};
const getUiReportFormat = backendField => {
  let res;
  for (let [ui, backend] of Object.entries(_backendModel.REPORT_FORMAT_DICT)) {
    if (backend === backendField) {
      res = ui;
    }
  }
  return res;
};
const getUiReportState = (status, delivery) => {
  let res;
  for (let [ui, backend] of Object.entries(_backendModel.REPORT_STATE_DICT)) {
    if (backend === status) {
      // distinguish "shared" and "created"
      if (status === _backendModel.BACKEND_REPORT_STATE.success && delivery) {
        res = _constants.REPORT_STATE.shared;
      } else {
        res = ui;
      }
    } else if (status === _backendModel.BACKEND_REPORT_STATE.scheduled) {
      // corner case
      res = _constants.REPORT_STATE.pending;
    }
  }
  return res;
};
const getUiReportSource = type => {
  let res;
  for (let [ui, backend] of Object.entries(_backendModel.REPORT_SOURCE_DICT)) {
    if (backend === type) {
      res = ui;
    }
  }
  return res;
};
const getUiReportDefinitionStatus = isEnabled => {
  return isEnabled ? _constants.REPORT_DEFINITION_STATUS.active : _constants.REPORT_DEFINITION_STATUS.disabled;
};
const getUiTriggerParams = (triggerType, schedule, createdTimeMs, isEnabled) => {
  let res = {
    trigger_type: getUiTriggerType(triggerType),
    ...(getUiTriggerType(triggerType) === _constants.TRIGGER_TYPE.schedule && getUiScheduleParams(schedule, createdTimeMs, isEnabled))
  };
  return res;
};

// Delivery
const getUiDeliveryParams = delivery => {
  const opensearchDashboardsUserDeliveryParams = {
    configIds: [],
    title: '',
    textDescription: '',
    htmlDescription: ''
  };
  let params;
  if (delivery) {
    const {
      ...rest
    } = delivery;
    params = {
      ...rest
    };
  } else {
    params = opensearchDashboardsUserDeliveryParams;
  }
  return params;
};

// helper function to add tenant info to url(if tenant is available)
const addTenantToURL = (url, userRequestedTenant) => {
  // build fake url from relative url
  const fakeUrl = `http://opensearch.com${url}`;
  const tenantKey = 'security_tenant';
  const tenantKeyAndValue = tenantKey + '=' + encodeURIComponent(userRequestedTenant);
  const {
    pathname,
    search
  } = (0, _url.parse)(fakeUrl);
  const queryDelimiter = !search ? '?' : '&';

  // The url parser returns null if the search is empty. Change that to an empty
  // string so that we can use it to build the values later
  if (search && search.toLowerCase().indexOf(tenantKey) > -1) {
    // If we for some reason already have a tenant in the URL we skip any updates
    return url;
  }

  // A helper for finding the part in the string that we want to extend/replace
  const valueToReplace = pathname + (search || '');
  const replaceWith = valueToReplace + queryDelimiter + tenantKeyAndValue;
  return url.replace(valueToReplace, replaceWith);
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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